import serial
from collections import deque
import numpy as np
import threading
import matplotlib.pyplot as plt
import matplotlib.animation as animation
import argparse


class SerialPlotter:
    def __init__(self, portNumber, length, windowSize=50):
        self.ser = serial.Serial(portNumber, 9600)
        self.length = length
        self.data = deque(length * [0])
        self.rms = deque(length * [0])
        self.rmsCounter = 0
        self.windowSize = windowSize
        self.startReading()

    # add to internal array
    def addSample(self, sample):
        if len(self.data) == self.length:
            self.data.pop()
            self.data.appendleft(sample)
        else:
            self.data.appendleft(sample)

    def calculateRms(self):
        temp = np.subtract(np.array(self.data), 512)
        temp = np.square(temp)
        temp = np.mean(temp.reshape(-1, self.windowSize), axis=1)
        temp = np.sqrt(temp)
        self.rms = [rmsValue for rmsValue in temp
                    for r in range(self.windowSize)]

    def readFromSerialPort(self):
        while True:
            try:
                line = self.ser.readline()
                sample = int(line)
                if sample > 0 and sample < 1024:
                    self.addSample(sample)
                    self.rmsCounter += 1
                    if self.rmsCounter >= self.windowSize:
                        self.calculateRms()
                        self.rmsCounter = 0
            except ValueError:
                print("TransmissionError")

    def close(self):
        self.ser.flush()
        self.ser.close()

    def startReading(self):
        t = threading.Thread(target=self.readFromSerialPort)
        t.daemon = True
        t.start()

    def updatePlot(self, frameNum, rawData, rms):
        try:
            rawData.set_data(range(self.length - 1, -1, -1), self.data)
            rms.set_data(range(self.length - 1, -1, -1), self.rms)
        except KeyboardInterrupt:
            print("Exiting!")
        return (rawData,rms)


def main():

    parser = argparse.ArgumentParser(description='Plotter for EMG samples read out from a serial port.')
    parser.add_argument("port", type=str, help='serial port to read from')

    args = parser.parse_args()

    plotter = SerialPlotter(args.port, 1000)

    #plotter = SerialPlotter("/dev/rfcomm1", 1000)
    print("Plotting data...")

    fig = plt.figure()
    ax = fig.add_subplot(2,1,1)
    bx = fig.add_subplot(2,1,2)

    ax.axis([0,999,0,1023])
    bx.axis([0,999,0,1023])

    rawData, = ax.plot([], [])
    rms, = bx.plot([], [])
    anim = animation.FuncAnimation(fig, plotter.updatePlot,
                                   fargs=(rawData,rms),
                                   interval=25, blit=True)
    plt.show()
    plotter.close()
    print("Exiting!")


if __name__ == '__main__':
    main()
